"""Application routes for authentication, admin, and user flows."""
from __future__ import annotations

import random
from collections import defaultdict
from datetime import datetime, timedelta
from typing import Any, Optional

import io
import json

from flask import (
    Blueprint,
    abort,
    flash,
    redirect,
    render_template,
    request,
    send_file,
    jsonify,
    url_for,
)
from flask_login import current_user, login_required, login_user, logout_user
from werkzeug.utils import secure_filename

from . import db
from .decorators import admin_required
from .forms import (
    AdminPasswordResetForm,
    LoginForm,
    PasswordChangeForm,
    TestAssignmentForm,
    TestForm,
    TestImportForm,
    TestStartForm,
    TestSubmitForm,
    UserCreateForm,
    UserProfileForm,
    UserUpdateForm,
)
from .models import (
    Answer,
    GradeCriteria,
    Question,
    Test,
    TestAssignment,
    TestAttempt,
    User,
    UserAnswer,
)
from .documents import (
    DocumentParseError,
    build_test_from_parsed,
    export_test_to_docx,
    parse_test_from_docx,
)

auth_bp = Blueprint("auth", __name__)
user_bp = Blueprint("user", __name__)
admin_bp = Blueprint("admin", __name__, url_prefix="/admin")


# ---------- Authentication ----------


@auth_bp.route("/login", methods=["GET", "POST"])
def login():
    if current_user.is_authenticated:
        return redirect(url_for("user.dashboard"))

    form = LoginForm()
    if form.validate_on_submit():
        user = User.query.filter_by(username=form.username.data).first()
        if user and user.check_password(form.password.data):
            login_user(user, remember=form.remember_me.data)
            flash("Добро пожаловать!", "success")
            next_page = request.args.get("next")
            if next_page:
                return redirect(next_page)
            if user.is_admin:
                return redirect(url_for("admin.dashboard"))
            return redirect(url_for("user.dashboard"))
        flash("Неверное имя пользователя или пароль", "danger")
    return render_template("auth/login.html", form=form)


@auth_bp.route("/logout")
@login_required
def logout():
    logout_user()
    flash("Вы вышли из системы", "info")
    return redirect(url_for("auth.login"))


# ---------- User routes ----------


@user_bp.route("/")
def index():
    if current_user.is_authenticated:
        if current_user.is_admin:
            return redirect(url_for("admin.dashboard"))
        return redirect(url_for("user.dashboard"))
    return redirect(url_for("auth.login"))


@user_bp.route("/dashboard")
@login_required
def dashboard():
    assigned_tests = _get_assigned_tests(current_user)
    completed_attempts = (
        TestAttempt.query.filter_by(user_id=current_user.id)
        .filter(TestAttempt.completed_at.isnot(None))
        .order_by(TestAttempt.completed_at.desc())
        .all()
    )
    average_score = None
    if completed_attempts:
        scores = [attempt.score for attempt in completed_attempts if attempt.score]
        if scores:
            average_score = round(sum(scores) / len(scores), 2)
    stats = {
        "taken": len(completed_attempts),
        "average_score": average_score,
    }
    return render_template(
        "user/dashboard.html",
        assigned_tests=assigned_tests,
        attempts=completed_attempts,
        stats=stats,
    )


@user_bp.route("/profile", methods=["GET", "POST"])
@login_required
def profile():
    password_form = PasswordChangeForm(prefix="password")
    profile_form = UserProfileForm(obj=current_user, prefix="profile")
    if request.method == "POST":
        form_name = request.form.get("form-name")
        if form_name == "profile" and profile_form.validate_on_submit():
            current_user.last_name = profile_form.last_name.data
            current_user.first_name = profile_form.first_name.data
            current_user.middle_name = profile_form.middle_name.data or None
            current_user.department = profile_form.department.data
            db.session.commit()
            flash("Профиль обновлен", "success")
            return redirect(url_for("user.profile"))
        if form_name == "password" and password_form.validate_on_submit():
            if not current_user.check_password(password_form.current_password.data):
                flash("Текущий пароль неверен", "danger")
            else:
                current_user.set_password(password_form.new_password.data)
                db.session.commit()
                flash("Пароль обновлен", "success")
                return redirect(url_for("user.profile"))

    attempts = (
        TestAttempt.query.filter_by(user_id=current_user.id)
        .filter(TestAttempt.completed_at.isnot(None))
        .order_by(TestAttempt.completed_at.desc())
        .all()
    )
    return render_template(
        "user/profile.html",
        password_form=password_form,
        profile_form=profile_form,
        attempts=attempts,
    )


@user_bp.route("/test/<int:test_id>", methods=["GET", "POST"])
@login_required
def test_detail(test_id: int):
    test = Test.query.get_or_404(test_id)
    if not _is_test_available(test, current_user):
        abort(403)

    active_attempt = (
        TestAttempt.query.filter_by(user_id=current_user.id, test_id=test.id)
        .filter(TestAttempt.completed_at.is_(None))
        .order_by(TestAttempt.started_at.desc())
        .first()
    )
    if active_attempt and active_attempt.expires_at and datetime.utcnow() > active_attempt.expires_at:
        _expire_attempt(active_attempt)
        db.session.commit()
        flash("Время на предыдущую попытку истекло.", "warning")
        return redirect(url_for("user.attempt_results", attempt_id=active_attempt.id))
    if active_attempt:
        return redirect(url_for("user.test_attempt", test_id=test.id, attempt_id=active_attempt.id))

    completed_attempts = (
        TestAttempt.query.filter_by(user_id=current_user.id, test_id=test.id)
        .filter(TestAttempt.completed_at.isnot(None))
        .count()
    )

    form = TestStartForm()
    if form.validate_on_submit():
        expires_at = None
        if test.time_limit_minutes:
            expires_at = datetime.utcnow() + timedelta(minutes=test.time_limit_minutes)
        questions = list(test.questions)
        # Сохраняем порядок вопросов, чтобы повторные запросы использовали ту же последовательность
        question_ids = [question.id for question in questions]
        if test.shuffle_questions:
            question_ids = list(question_ids)
            random.shuffle(question_ids)

        answer_order: dict[str, list[int]] = {}
        for question in questions:
            answer_ids = [answer.id for answer in question.answers]
            if test.shuffle_answers:
                answer_ids = list(answer_ids)
                random.shuffle(answer_ids)
            # Фиксируем порядок вариантов для каждого вопроса
            answer_order[str(question.id)] = answer_ids
        attempt = TestAttempt(
            user_id=current_user.id,
            test_id=test.id,
            total_questions=len(test.questions),
            expires_at=expires_at,
            question_order=json.dumps(question_ids),
            answer_order=json.dumps(answer_order),
        )
        db.session.add(attempt)
        db.session.commit()
        return redirect(url_for("user.test_attempt", test_id=test.id, attempt_id=attempt.id))

    latest_attempt = (
        TestAttempt.query.filter_by(user_id=current_user.id, test_id=test.id)
        .order_by(TestAttempt.started_at.desc())
        .first()
    )

    return render_template(
        "user/test_detail.html",
        test=test,
        form=form,
        latest_attempt=latest_attempt,
        completed_attempts=completed_attempts,
    )


@user_bp.route("/test/<int:test_id>/attempt/<int:attempt_id>", methods=["GET", "POST"])
@login_required
def test_attempt(test_id: int, attempt_id: int):
    attempt = TestAttempt.query.get_or_404(attempt_id)
    test = Test.query.get_or_404(test_id)
    if attempt.user_id != current_user.id or attempt.test_id != test.id:
        abort(403)
    if attempt.expires_at and datetime.utcnow() > attempt.expires_at:
        _expire_attempt(attempt)
        db.session.commit()
        flash("Время на прохождение теста истекло.", "warning")
        return redirect(url_for("user.attempt_results", attempt_id=attempt.id))
    if attempt.completed_at:
        return redirect(url_for("user.attempt_results", attempt_id=attempt.id))

    question_payload = _build_question_payload(test, attempt)

    form = TestSubmitForm()
    if form.validate_on_submit():
        if attempt.expires_at and datetime.utcnow() > attempt.expires_at:
            _expire_attempt(attempt)
            db.session.commit()
            flash("Время на прохождение теста истекло.", "warning")
            return redirect(url_for("user.attempt_results", attempt_id=attempt.id))
        _persist_attempt_answers(attempt, test, request.form)
        db.session.commit()
        flash("Тест завершен", "success")
        return redirect(url_for("user.attempt_results", attempt_id=attempt.id))

    remaining_seconds = _calculate_remaining_seconds(attempt)
    expires_at_local = None
    if attempt.expires_at:
        expires_at_local = attempt.expires_at + timedelta(hours=3)
    return render_template(
        "user/attempt.html",
        test=test,
        attempt=attempt,
        question_payload=question_payload,
        form=form,
        time_limit_minutes=test.time_limit_minutes,
        remaining_seconds=remaining_seconds,
        remaining_time_display=_format_remaining_time(remaining_seconds),
        status_url=url_for("user.attempt_status", test_id=test.id, attempt_id=attempt.id),
        expires_at_local=expires_at_local,
    )


@user_bp.route("/test/<int:test_id>/attempt/<int:attempt_id>/status")
@login_required
def attempt_status(test_id: int, attempt_id: int):
    attempt = TestAttempt.query.get_or_404(attempt_id)
    if attempt.user_id != current_user.id or attempt.test_id != test_id:
        abort(403)
    if attempt.completed_at:
        return jsonify({"status": "completed", "remaining_seconds": 0, "completed": True})
    if attempt.expires_at and datetime.utcnow() > attempt.expires_at:
        _expire_attempt(attempt)
        db.session.commit()
        return jsonify({"status": "expired", "remaining_seconds": 0, "completed": True})
    remaining = _calculate_remaining_seconds(attempt)
    return jsonify({"status": "active", "remaining_seconds": remaining, "completed": False})


@user_bp.route("/attempt/<int:attempt_id>/results")
@login_required
def attempt_results(attempt_id: int):
    attempt = TestAttempt.query.get_or_404(attempt_id)
    if attempt.user_id != current_user.id and not current_user.is_admin:
        abort(403)
    return render_template("user/attempt_results.html", attempt=attempt)


# ---------- Admin routes ----------


@admin_bp.route("/dashboard")
@admin_required
def dashboard():
    user_count = User.query.count()
    test_count = Test.query.count()
    active_attempts = TestAttempt.query.filter(TestAttempt.completed_at.is_(None)).count()
    recent_attempts = (
        TestAttempt.query.filter(TestAttempt.completed_at.isnot(None))
        .order_by(TestAttempt.completed_at.desc())
        .limit(5)
        .all()
    )
    return render_template(
        "admin/dashboard.html",
        user_count=user_count,
        test_count=test_count,
        active_attempts=active_attempts,
        recent_attempts=recent_attempts,
    )


@admin_bp.route("/users", methods=["GET", "POST"])
@admin_required
def manage_users():
    form = UserCreateForm()
    if form.validate_on_submit():
        if User.query.filter_by(username=form.username.data).first():
            flash("Пользователь с таким именем уже существует", "danger")
        else:
            user = User(
                username=form.username.data,
                last_name=form.last_name.data,
                first_name=form.first_name.data,
                middle_name=form.middle_name.data or None,
                department=form.department.data,
                role=form.role.data,
            )
            user.set_password(form.password.data)
            db.session.add(user)
            db.session.commit()
            flash("Пользователь создан", "success")
            return redirect(url_for("admin.manage_users"))

    users = User.query.order_by(User.created_at.desc()).all()
    stats = _build_user_statistics(users)
    return render_template("admin/users.html", form=form, users=users, stats=stats)


@admin_bp.route("/users/<int:user_id>", methods=["GET", "POST"])
@admin_required
def edit_user(user_id: int):
    user = User.query.get_or_404(user_id)
    form = UserUpdateForm(obj=user)
    reset_form = AdminPasswordResetForm()

    if request.method == "POST":
        form_name = request.form.get("form-name")
        if form_name == "user-update" and form.validate_on_submit():
            user.last_name = form.last_name.data
            user.first_name = form.first_name.data
            user.middle_name = form.middle_name.data or None
            user.department = form.department.data
            user.role = form.role.data
            db.session.commit()
            flash("Данные пользователя обновлены", "success")
            return redirect(url_for("admin.edit_user", user_id=user.id))
        if form_name == "reset-password" and reset_form.validate_on_submit():
            user.set_password(reset_form.new_password.data)
            db.session.commit()
            flash("Пароль сброшен", "success")
            return redirect(url_for("admin.edit_user", user_id=user.id))

    attempts = (
        TestAttempt.query.filter_by(user_id=user.id)
        .filter(TestAttempt.completed_at.isnot(None))
        .order_by(TestAttempt.completed_at.desc())
        .all()
    )
    assigned_tests: list[dict[str, Any]] = []
    for test in _get_assigned_tests(user):
        relevant_assignments = [
            assignment
            for assignment in test.assignments
            if assignment.assigned_to_all or assignment.user_id == user.id
        ]
        scope = "all" if any(a.assigned_to_all for a in relevant_assignments) else "user"
        assigned_at = max((assignment.assigned_at for assignment in relevant_assignments), default=None)
        attempts_query = (
            TestAttempt.query.filter_by(user_id=user.id, test_id=test.id)
            .order_by(TestAttempt.completed_at.desc())
        )
        completed_attempt = attempts_query.filter(TestAttempt.completed_at.isnot(None))
        attempts_count = completed_attempt.count()
        latest_completed_attempt = completed_attempt.first()
        latest_score = latest_completed_attempt.score if latest_completed_attempt else None
        last_completed_at = latest_completed_attempt.completed_at if latest_completed_attempt else None
        assigned_tests.append(
            {
                "test": test,
                "scope": scope,
                "assigned_at": assigned_at,
                "attempts_count": attempts_count,
                "last_completed_at": last_completed_at,
                "latest_score": latest_score,
            }
        )
    return render_template(
        "admin/user_detail.html",
        user=user,
        form=form,
        reset_form=reset_form,
        attempts=attempts,
        assigned_tests=assigned_tests,
    )


@admin_bp.route("/tests")
@admin_required
def manage_tests():
    tests = Test.query.order_by(Test.created_at.desc()).all()
    return render_template("admin/tests.html", tests=tests)


@admin_bp.route("/test/import", methods=["GET", "POST"])
@admin_required
def import_test():
    form = TestImportForm()
    if form.validate_on_submit():
        file_storage = form.document.data
        file_bytes = file_storage.read()
        buffer = io.BytesIO(file_bytes)
        try:
            parsed_test = parse_test_from_docx(buffer)
            test = build_test_from_parsed(parsed_test, created_by=current_user.id)
        except DocumentParseError as exc:
            flash(str(exc), "danger")
            return render_template("admin/test_import.html", form=form)

        if Test.query.filter_by(title=test.title).first():
            flash("Тест с таким названием уже существует", "danger")
            return render_template("admin/test_import.html", form=form)

        db.session.add(test)
        db.session.commit()
        flash("Тест импортирован", "success")
        return redirect(url_for("admin.manage_tests"))
    return render_template("admin/test_import.html", form=form)


@admin_bp.route("/test/<int:test_id>/export")
@admin_required
def export_test(test_id: int):
    test = Test.query.get_or_404(test_id)
    stream = export_test_to_docx(test)
    filename_base = secure_filename(test.title) or f"test-{test.id}"
    download_name = f"{filename_base}.docx"
    return send_file(
        stream,
        mimetype="application/vnd.openxmlformats-officedocument.wordprocessingml.document",
        as_attachment=True,
        download_name=download_name,
    )


@admin_bp.route("/test/create", methods=["GET", "POST"])
@admin_required
def create_test():
    form = TestForm()
    if request.method == "GET":
        _ensure_form_minimums(form)

    if form.is_submitted() and _handle_dynamic_form_actions(form):
        return render_template("admin/test_form.html", form=form, mode="create")

    if form.validate_on_submit():
        test = Test(
            title=form.title.data,
            description=form.description.data,
            created_by=current_user.id,
            is_active=form.is_active.data,
            shuffle_questions=form.shuffle_questions.data,
            shuffle_answers=form.shuffle_answers.data,
            time_limit_minutes=form.time_limit_minutes.data or None,
        )
        success, error = _apply_questions_to_test(test, form.questions.entries)
        if not success:
            flash(error or "Проверьте корректность вопросов и ответов.", "danger")
            return render_template("admin/test_form.html", form=form, mode="create")
        success, error = _apply_criteria_to_test(test, form.criteria.entries)
        if not success:
            flash(error or "Проверьте настройки критериев оценивания.", "danger")
            return render_template("admin/test_form.html", form=form, mode="create")

        db.session.add(test)
        db.session.commit()
        flash("Тест создан", "success")
        return redirect(url_for("admin.manage_tests"))

    return render_template("admin/test_form.html", form=form, mode="create")


@admin_bp.route("/test/<int:test_id>/edit", methods=["GET", "POST"])
@admin_required
def edit_test(test_id: int):
    test = Test.query.get_or_404(test_id)
    form = TestForm(obj=test)
    if request.method == "GET":
        while len(form.questions.entries):
            form.questions.pop_entry()
        for question in test.questions:
            question_entry = form.questions.append_entry()
            question_form = question_entry.form
            question_form.question_text.data = question.question_text
            question_form.question_type.data = question.question_type
            while len(question_form.answers.entries):
                question_form.answers.pop_entry()
            for answer in question.answers:
                answer_entry = question_form.answers.append_entry()
                answer_form = answer_entry.form
                answer_form.answer_text.data = answer.answer_text
                answer_form.is_correct.data = answer.is_correct
        while len(form.criteria.entries):
            form.criteria.pop_entry()
        for criteria in test.grade_criteria:
            criteria_entry = form.criteria.append_entry()
            criteria_form = criteria_entry.form
            criteria_form.grade.data = str(criteria.grade)
            criteria_form.max_errors.data = criteria.max_errors
        _ensure_form_minimums(form)

    if form.is_submitted() and _handle_dynamic_form_actions(form):
        return render_template("admin/test_form.html", form=form, mode="edit", test=test)

    if form.validate_on_submit():
        test.title = form.title.data
        test.description = form.description.data
        test.is_active = form.is_active.data
        test.shuffle_questions = form.shuffle_questions.data
        test.shuffle_answers = form.shuffle_answers.data
        test.time_limit_minutes = form.time_limit_minutes.data or None

        test.questions.clear()
        test.grade_criteria.clear()

        success, error = _apply_questions_to_test(test, form.questions.entries)
        if not success:
            flash(error or "Проверьте корректность вопросов и ответов.", "danger")
            return render_template("admin/test_form.html", form=form, mode="edit", test=test)
        success, error = _apply_criteria_to_test(test, form.criteria.entries)
        if not success:
            flash(error or "Проверьте настройки критериев оценивания.", "danger")
            return render_template("admin/test_form.html", form=form, mode="edit", test=test)

        db.session.commit()
        flash("Тест обновлен", "success")
        return redirect(url_for("admin.manage_tests"))

    return render_template("admin/test_form.html", form=form, mode="edit", test=test)


@admin_bp.route("/test/<int:test_id>/delete", methods=["POST"])
@admin_required
def delete_test(test_id: int):
    test = Test.query.get_or_404(test_id)
    db.session.delete(test)
    db.session.commit()
    flash("Тест удален", "info")
    return redirect(url_for("admin.manage_tests"))


@admin_bp.route("/test/<int:test_id>/assign", methods=["GET", "POST"])
@admin_required
def assign_test(test_id: int):
    test = Test.query.get_or_404(test_id)
    form = TestAssignmentForm()
    users = User.query.order_by(User.last_name, User.first_name, User.username).all()
    form.users.choices = [
        (
            user.id,
            user.full_name if user.full_name else user.username,
        )
        for user in users
    ]
    department_map: dict[str, list[User]] = {}
    for user in users:
        if user.department:
            department_map.setdefault(user.department, []).append(user)
    # Собираем краткую справку по отделам для отображения и множественного выбора
    department_info = [
        {"name": name, "count": len(members)}
        for name, members in sorted(department_map.items(), key=lambda item: item[0].lower())
    ]
    form.departments.choices = [(info["name"], info["name"]) for info in department_info]

    if form.validate_on_submit():
        # Clear existing assignments
        TestAssignment.query.filter_by(test_id=test.id).delete()
        db.session.flush()

        if form.assigned_to_all.data:
            assignment = TestAssignment(test_id=test.id, assigned_to_all=True)
            db.session.add(assignment)
        else:
            selected_user_ids: set[int] = set(form.users.data or [])
            selected_departments = form.departments.data or []
            for department_name in selected_departments:
                for member in department_map.get(department_name, []):
                    # Добавляем всех сотрудников выбранного отдела
                    selected_user_ids.add(member.id)

            if not selected_user_ids:
                flash("Выберите хотя бы одного пользователя или отметьте 'для всех'", "danger")
                return render_template(
                    "admin/test_assign.html",
                    form=form,
                    test=test,
                    users=users,
                    departments=department_info,
                )
            for user_id in sorted(selected_user_ids):
                db.session.add(
                    TestAssignment(test_id=test.id, user_id=user_id, assigned_to_all=False)
                )
        db.session.commit()
        flash("Назначение обновлено", "success")
        return redirect(url_for("admin.manage_tests"))

    existing_all = TestAssignment.query.filter_by(test_id=test.id, assigned_to_all=True).first()
    form.assigned_to_all.data = bool(existing_all)
    if existing_all:
        form.users.data = [user.id for user in users]
        form.departments.data = [info["name"] for info in department_info]
    else:
        form.users.data = [
            assignment.user_id
            for assignment in test.assignments
            if assignment.user_id is not None
        ]

    return render_template(
        "admin/test_assign.html",
        form=form,
        test=test,
        users=users,
        departments=department_info,
    )
@admin_bp.route("/attempt/<int:attempt_id>/report")
@admin_required
def attempt_report(attempt_id: int):
    attempt = TestAttempt.query.get_or_404(attempt_id)
    test = attempt.test
    question_payload = _build_question_payload(test, attempt)

    answers_by_question: dict[int, list[UserAnswer]] = defaultdict(list)
    for answer in attempt.answers:
        answers_by_question[answer.question_id].append(answer)

    report_rows: list[dict[str, Any]] = []
    for question, answers in question_payload:
        user_answers = answers_by_question.get(question.id, [])
        selected_ids = {
            answer.selected_answer_id
            for answer in user_answers
            if answer.selected_answer_id is not None
        }
        correct_ids = {answer.id for answer in answers if answer.is_correct}
        is_correct = bool(correct_ids) and selected_ids == correct_ids
        unanswered = not selected_ids and not any(
            answer.selected_answer_id for answer in user_answers
        )
        option_rows = []
        for answer in answers:
            # Для каждой опции фиксируем, выбрал ли её пользователь и является ли она верной
            option_rows.append(
                {
                    "answer": answer,
                    "is_correct": answer.is_correct,
                    "is_selected": answer.id in selected_ids,
                    "status": _determine_option_status(
                        answer.is_correct, answer.id in selected_ids
                    ),
                }
            )
        report_rows.append(
            {
                "question": question,
                "options": option_rows,
                "is_correct": is_correct,
                "unanswered": unanswered,
                "selected_ids": selected_ids,
                "correct_ids": correct_ids,
            }
        )

    total_correct = sum(1 for row in report_rows if row["is_correct"])
    return render_template(
        "admin/attempt_report.html",
        attempt=attempt,
        test=test,
        user=attempt.user,
        report_rows=report_rows,
        total_correct=total_correct,
        total_questions=len(report_rows),
    )


@admin_bp.route("/statistics")
@admin_required
def statistics():
    tests = Test.query.all()
    test_stats = []
    detailed_reports: dict[int, list[dict[str, Any]]] = {}

    for test in tests:
        attempts = [a for a in test.attempts if a.completed_at]
        if not attempts:
            test_stats.append(
                {
                    "test": test,
                    "attempts": 0,
                    "average_score": None,
                    "pass_rate": None,
                }
            )
            detailed_reports[test.id] = []
            continue

        completed_scores = [attempt.score for attempt in attempts if attempt.score]
        average_score = round(sum(completed_scores) / len(completed_scores), 2) if completed_scores else None
        pass_rate = round(
            len([attempt for attempt in attempts if attempt.score and attempt.score >= 3]) / len(attempts) * 100,
            2,
        )
        test_stats.append(
            {
                "test": test,
                "attempts": len(attempts),
                "average_score": average_score,
                "pass_rate": pass_rate,
            }
        )

        user_attempts_map: dict[int, list[TestAttempt]] = {}
        for attempt in attempts:
            # Группируем попытки по пользователям, чтобы посчитать статистику по каждому
            user_attempts_map.setdefault(attempt.user_id, []).append(attempt)

        report_rows: list[dict[str, Any]] = []
        for user_id, attempts_list in user_attempts_map.items():
            attempts_list.sort(key=lambda a: a.started_at)
            last_attempt = max(attempts_list, key=lambda a: a.completed_at or a.started_at)
            user = last_attempt.user
            report_rows.append(
                {
                    "user": user,
                    "attempt_count": len(attempts_list),
                    "last_completed_at": last_attempt.completed_at,
                    "last_score": last_attempt.score,
                    "last_attempt_id": last_attempt.id,
                }
            )

        report_rows.sort(key=lambda row: (row["last_completed_at"] or row["user"].created_at), reverse=True)
        detailed_reports[test.id] = report_rows

    user_attempts = (
        TestAttempt.query.filter(TestAttempt.completed_at.isnot(None))
        .order_by(TestAttempt.completed_at.desc())
        .limit(20)
        .all()
    )
    return render_template(
        "admin/statistics.html",
        test_stats=test_stats,
        user_attempts=user_attempts,
        detailed_reports=detailed_reports,
    )


# ---------- Helper functions ----------


def _get_assigned_tests(user: User) -> list[Test]:
    assigned = Test.query.filter_by(is_active=True).all()
    visible_tests = []
    for test in assigned:
        assignments = list(test.assignments)
        if not assignments:
            continue
        if any(assignment.assigned_to_all for assignment in assignments):
            visible_tests.append(test)
        elif any(assignment.user_id == user.id for assignment in assignments):
            visible_tests.append(test)
    return visible_tests


def _is_test_available(test: Test, user: User) -> bool:
    if not test.is_active:
        return False
    assignments = list(test.assignments)
    if not assignments:
        return False
    return any(
        assignment.assigned_to_all or assignment.user_id == user.id for assignment in assignments
    )


def _build_question_payload(test: Test, attempt: TestAttempt) -> list[tuple[Question, list[Answer]]]:
    # Загружаем вопросы, сохраняя порядок, который был закреплён при запуске попытки
    questions_by_id: dict[int, Question] = {question.id: question for question in test.questions}

    question_ids = [question.id for question in test.questions]
    if attempt.question_order:
        try:
            stored_ids = [int(value) for value in json.loads(attempt.question_order)]
            ordered_ids = [qid for qid in stored_ids if qid in questions_by_id]
            question_ids = ordered_ids + [qid for qid in questions_by_id if qid not in ordered_ids]
        except (ValueError, TypeError):
            pass

    answer_order_map: dict[str, list[int]] = {}
    if attempt.answer_order:
        try:
            parsed = json.loads(attempt.answer_order)
            if isinstance(parsed, dict):
                answer_order_map = parsed
        except (ValueError, TypeError):
            pass

    payload: list[tuple[Question, list[Answer]]] = []
    for question_id in question_ids:
        question = questions_by_id.get(question_id)
        if not question:
            continue
        answers = list(question.answers)
        key = str(question_id)
        if key in answer_order_map:
            desired_order = answer_order_map[key]
            answers_by_id: dict[int, Answer] = {answer.id: answer for answer in answers}
            ordered_answers: list[Answer] = []
            for ans_id in desired_order:
                answer = answers_by_id.get(ans_id)
                if answer:
                    ordered_answers.append(answer)
            ordered_ids = {answer.id for answer in ordered_answers}
            remaining_answers = [answer for answer in answers if answer.id not in ordered_ids]
            answers = ordered_answers + remaining_answers
        payload.append((question, answers))

    return payload


def _persist_attempt_answers(attempt: TestAttempt, test: Test, form_data) -> None:
    UserAnswer.query.filter_by(attempt_id=attempt.id).delete()
    db.session.flush()

    total_questions = len(test.questions)
    correct_answers = 0
    for question in test.questions:
        field_name = f"question-{question.id}"
        submitted = form_data.getlist(field_name)
        submitted_ids = {int(value) for value in submitted if value.isdigit()}

        correct_ids = {answer.id for answer in question.answers if answer.is_correct}
        is_correct = False
        if question.question_type == "single":
            if submitted_ids:
                selected_id = next(iter(submitted_ids))
                is_correct = selected_id in correct_ids
                db.session.add(
                    UserAnswer(
                        attempt_id=attempt.id,
                        question_id=question.id,
                        selected_answer_id=selected_id,
                        is_correct=is_correct,
                    )
                )
        else:
            is_correct = submitted_ids == correct_ids and bool(submitted_ids)
            for answer_id in submitted_ids:
                db.session.add(
                    UserAnswer(
                        attempt_id=attempt.id,
                        question_id=question.id,
                        selected_answer_id=answer_id,
                        is_correct=answer_id in correct_ids,
                    )
                )

        if is_correct:
            correct_answers += 1
        else:
            # Add placeholder so we have a record of unanswered questions
            if not submitted_ids:
                db.session.add(
                    UserAnswer(
                        attempt_id=attempt.id,
                        question_id=question.id,
                        selected_answer_id=None,
                        is_correct=False,
                    )
                )

    errors = total_questions - correct_answers
    score = test.grade_for_errors(errors)
    attempt.mark_completed(score=score, correct_answers=correct_answers, total_questions=total_questions)


def _apply_questions_to_test(test: Test, question_entries) -> tuple[bool, str | None]:
    if not question_entries:
        return False, "Добавьте хотя бы один вопрос."

    prepared_questions: list[Question] = []
    for index, question_entry in enumerate(question_entries, start=1):
        form = question_entry.form
        answers = [
            answer_entry.form
            for answer_entry in form.answers.entries
            if answer_entry.form.answer_text.data
        ]
        if len(answers) < 2:
            return False, f"Вопрос №{index} должен иметь минимум два варианта ответа."

        correct_answers = [answer_form for answer_form in answers if answer_form.is_correct.data]
        if not correct_answers:
            return False, f"Укажите хотя бы один правильный ответ для вопроса №{index}."

        if form.question_type.data == "single" and len(correct_answers) != 1:
            return False, f"Для вопроса №{index} с одиночным выбором допустим только один правильный ответ."

        question = Question(
            question_text=form.question_text.data,
            question_type=form.question_type.data,
        )
        for answer_form in answers:
            question.answers.append(
                Answer(
                    answer_text=answer_form.answer_text.data,
                    is_correct=bool(answer_form.is_correct.data),
                )
            )
        prepared_questions.append(question)

    if not prepared_questions:
        return False, "Не удалось сохранить вопросы теста."

    test.questions.extend(prepared_questions)
    return True, None


def _apply_criteria_to_test(test: Test, criteria_entries) -> tuple[bool, str | None]:
    expected_grades = [5, 4, 3]
    if not criteria_entries or len(criteria_entries) < len(expected_grades):
        return False, "Укажите параметры оценивания для оценок 5, 4 и 3."

    grade_to_form: dict[int, Any] = {}
    for criteria_entry in criteria_entries:
        try:
            grade_value = int(criteria_entry.form.grade.data)
        except (TypeError, ValueError):
            continue
        grade_to_form[grade_value] = criteria_entry.form

    if not all(grade in grade_to_form for grade in expected_grades):
        return False, "Не удалось определить настройки для всех оценок 5, 4 и 3."

    max_errors_values: list[int] = []
    for grade in expected_grades:
        form = grade_to_form[grade]
        value = form.max_errors.data
        if value in (None, ""):
            return False, "Заполните количество допустимых ошибок для каждого критерия."
        try:
            max_errors = int(value)
        except (TypeError, ValueError):
            return False, "Количество ошибок должно быть целым неотрицательным числом."
        if max_errors < 0:
            return False, "Количество ошибок не может быть отрицательным."
        max_errors_values.append(max_errors)

    if not (max_errors_values[0] <= max_errors_values[1] <= max_errors_values[2]):
        return False, "Количество ошибок для оценки 5 должно быть не больше, чем для 4, а для 4 — не больше, чем для 3."

    prepared_criteria = [
        GradeCriteria(grade=grade, max_errors=max_errors)
        for grade, max_errors in zip(expected_grades, max_errors_values)
    ]

    test.grade_criteria.extend(prepared_criteria)
    return True, None


def _ensure_form_minimums(form: TestForm) -> None:
    if len(form.questions.entries) == 0:
        question_entry = form.questions.append_entry()
        question_form = question_entry.form
        question_form.question_type.data = "single"

    for question_entry in form.questions.entries:
        question_form = question_entry.form
        if not question_form.question_type.data:
            question_form.question_type.data = "single"
        answers_field = question_form.answers
        existing_answers = len(answers_field.entries)
        minimum_required = 4 if existing_answers == 0 else 2
        while len(answers_field.entries) < minimum_required:
            answer_entry = answers_field.append_entry()
            answer_entry.form.is_correct.data = False

    required_grades = [5, 4, 3]
    existing_values: dict[int, Any] = {}
    for criteria_entry in form.criteria.entries:
        try:
            grade_value = int(criteria_entry.form.grade.data)
        except (TypeError, ValueError):
            continue
        existing_values[grade_value] = criteria_entry.form.max_errors.data

    while len(form.criteria.entries):
        form.criteria.pop_entry()

    for grade in required_grades:
        criteria_entry = form.criteria.append_entry()
        criteria_form = criteria_entry.form
        criteria_form.grade.data = str(grade)
        stored_value = existing_values.get(grade)
        if stored_value in (None, ""):
            criteria_form.max_errors.data = 0
        else:
            try:
                criteria_form.max_errors.data = int(stored_value)
            except (TypeError, ValueError):
                criteria_form.max_errors.data = 0


def _handle_dynamic_form_actions(form: TestForm) -> bool:
    action_performed = False

    if "add-question" in request.form:
        question_entry = form.questions.append_entry()
        question_form = question_entry.form
        question_form.question_type.data = "single"
        action_performed = True
    else:
        for question_index, question_entry in enumerate(form.questions.entries):
            answers_field = question_entry.form.answers
            if f"add-answer-{question_index}" in request.form:
                answer_entry = answers_field.append_entry()
                answer_entry.form.is_correct.data = False
                action_performed = True
                break
            if f"remove-question-{question_index}" in request.form:
                if len(form.questions.entries) > 1:
                    form.questions.entries.pop(question_index)
                else:
                    question_form = question_entry.form
                    question_form.question_text.data = ""
                    question_form.question_type.data = "single"
                    while len(answers_field.entries):
                        answers_field.pop_entry()
                    for _ in range(4):
                        answer_entry = answers_field.append_entry()
                        answer_entry.form.answer_text.data = ""
                        answer_entry.form.is_correct.data = False
                action_performed = True
                break
            for answer_index, _ in enumerate(list(answers_field.entries)):
                if f"remove-answer-{question_index}-{answer_index}" in request.form:
                    if len(answers_field.entries) > 2:
                        answers_field.entries.pop(answer_index)
                    else:
                        answer_form = answers_field.entries[answer_index].form
                        answer_form.answer_text.data = ""
                        answer_form.is_correct.data = False
                    action_performed = True
                    break
            if action_performed:
                break

    if action_performed:
        _ensure_form_minimums(form)
    return action_performed
def _expire_attempt(attempt: TestAttempt) -> None:
    if attempt.completed_at:
        return
    attempt.completed_at = attempt.expires_at or datetime.utcnow()
    attempt.score = 2
    if attempt.total_questions == 0 and attempt.test:
        attempt.total_questions = len(attempt.test.questions)
    attempt.correct_answers = attempt.correct_answers or 0


def _calculate_remaining_seconds(attempt: TestAttempt) -> Optional[int]:
    if not attempt.expires_at:
        return None
    remaining = int((attempt.expires_at - datetime.utcnow()).total_seconds())
    return remaining if remaining > 0 else 0


def _format_remaining_time(seconds: Optional[int]) -> str:
    if seconds is None:
        return ""
    minutes, secs = divmod(seconds, 60)
    hours, minutes = divmod(minutes, 60)
    parts = []
    if hours:
        parts.append(f"{hours} ч")
    if minutes:
        parts.append(f"{minutes} мин")
    if not parts:
        parts.append("менее минуты")
    if secs and hours == 0:
        parts.append(f"{secs} сек")
    return " ".join(parts)


def _determine_option_status(is_correct: bool, is_selected: bool) -> str:
    """Возвращает метку для подсветки варианта ответа в отчёте."""
    if is_correct and is_selected:
        return "correct-selected"
    if is_correct:
        return "correct-missed"
    if is_selected:
        return "incorrect-selected"
    return "incorrect"


def _build_user_statistics(users: list[User]) -> dict[int, dict[str, float | int | None]]:
    stats: dict[int, dict[str, float | int | None]] = defaultdict(dict)
    for user in users:
        attempts = (
            TestAttempt.query.filter_by(user_id=user.id)
            .filter(TestAttempt.completed_at.isnot(None))
            .all()
        )
        if not attempts:
            stats[user.id] = {"attempts": 0, "average_score": None}
            continue
        scores = [attempt.score for attempt in attempts if attempt.score]
        average = round(sum(scores) / len(scores), 2) if scores else None
        stats[user.id] = {"attempts": len(attempts), "average_score": average}
    return stats
